/*
 * FILE: A3DParticleEmitter.h
 *
 * DESCRIPTION: Abstract interface for Particle Emitter
 *
 * CREATED BY: ZhangYu, 2004/8/3
 *
 * HISTORY:
 *
 */

#ifndef _A3DPARTICLEEMITTER_H_
#define _A3DPARTICLEEMITTER_H_

#include "A3DTypes.h"
#include "A3DParticle.h"
#include "A3DGFXEditorInterface.h"
#include "AFile.h"
#include "A3DFuncs.h"

#define		EMITTER_TYPE_NONE				-1
#define		EMITTER_TYPE_POINT				100
#define		EMITTER_TYPE_BOX				101
#define		EMITTER_TYPE_MULTIPLANE			102
#define		EMITTER_TYPE_ELLIPSOID			103
#define		EMITTER_TYPE_CYLINDER			104
#define		EMITTER_TYPE_CURVE				105

// #define		ID_GFXOP_EMITTER_POS		200
#define		ID_GFXOP_EMITTER_RATE			201
// #define		ID_GFXOP_EMITTER_DIR		202
#define		ID_GFXOP_EMITTER_ANGLE			203
#define		ID_GFXOP_EMITTER_SPEED			204
#define		ID_GFXOP_EMITTER_ACC_DIR		205
#define		ID_GFXOP_EMITTER_ACC			206
#define		ID_GFXOP_EMITTER_TTL			207
#define		ID_GFXOP_EMITTER_COLOR_MIN		208
#define		ID_GFXOP_EMITTER_COLOR_MAX		209
#define		ID_GFXOP_EMITTER_SCALE_MIN		210
#define		ID_GFXOP_EMITTER_SCALE_MAX		211
#define		ID_GFXOP_EMITTER_ROT_MIN		212
#define		ID_GFXOP_EMITTER_ROT_MAX		213
#define		ID_GFXOP_EMITTER_AREA_SIZE		214
#define		ID_GFXOP_EMITTER_SURFACE		215
#define		ID_GFXOP_EMITTER_BIND			216
#define		ID_GFXOP_EMITTER_DRAG			217
#define		ID_GFXOP_EMITTER_PAR_ACC		218
#define		ID_GFXOP_EMITTER_CURVE_R		219
#define		ID_GFXOP_EMITTER_DRAG_POWER		220
#define		ID_GFXOP_EMITTER_REF_MOX		221
#define		ID_GFXOP_EMITTER_USE_NORMAL		222
#define		ID_GFXOP_EMITTER_PERP_TO_NORMAL	223
#define		ID_GFXOP_EMITTER_PARTICAL_INITDIR		224
#define		ID_GFXOP_EMITTER_CURVE_ISAVG			225
#define		ID_GFXOP_EMITTER_CURVE_AVGSEGNUM		226
#define		ID_GFXOP_EMITTER_CURVE_AVGGENSEQUTIALLY 227
#define		ID_GFXOP_EMITTER_COLOR_HSVINTERP		228
#define		ID_GFXOP_EMITTER_ELLIPSOID_ISAVG		229
#define		ID_GFXOP_EMITTER_ELLIPSOID_ALPHASEG		230
#define		ID_GFXOP_EMITTER_ELLIPSOID_BETASEG		231
#define		ID_GFXOP_EMITTER_CYLINDER_ISAVG			232
#define		ID_GFXOP_EMITTER_CYLINDER_ALPHASEG		233
#define		ID_GFXOP_EMITTER_CYLINDER_BETASEG		234
#define		ID_GFXOP_EMITTER_REF_SMD		235
#define		ID_GFXOP_EMITTER_SMD_SKIN_INDEX		236
#define		ID_GFXOP_EMITTER_SMD_SKINMESH_INDEX		237

class A3DParticleEmitter
{
public:
	A3DParticleEmitter();
	virtual ~A3DParticleEmitter();

protected:
	// 
	int				m_nType;
	// 
	float			m_fEmissionRate;
	float			m_fRemainder;
	// طƫתķΧ
	float			m_fAngle;
	// ٶ
	float			m_fSpeed;
	// ٶ
	float			m_fParAcc;
	// ٶȷ
	A3DVECTOR3		m_vAccDir;
	// ٶ
	float			m_fAcc;
	// , Time to live
	float			m_fTTL;
	// ɫ
	A3DCOLOR		m_colorMin;
	A3DCOLOR		m_colorMax;
	// HSLɫ: ӦA3DCOLORcolorMin colorMaxΪʱݣ洢
	A3DHSVCOLORVALUE	m_hsvMin;
	A3DHSVCOLORVALUE	m_hsvMax;
	// ӱ
	float			m_fScaleMin;
	float			m_fScaleMax;
	// ӳʼתǶ
	float			m_fRotMin;
	float			m_fRotMax;
	// Parentλ
	A3DVECTOR3		m_vParentPos;
	// Parent
	A3DQUATERNION	m_qParentDir;
	// Ƿڱ
	bool			m_bSurface;
	bool			m_bBind;
	bool			m_bDrag;
	float			m_fDragIntensity;
	// alpha
	float			m_fEmitterAlpha;
	// Ĺؼ֡
	A3DGFXKeyPoint* m_pKeyPoint;
	
	//  
	typedef	 A3DVECTOR3	EULERANGLE3;
	EULERANGLE3		m_vParticalInitDir;
	A3DQUATERNION	m_qParticleDir;

	// ǷʹHSVֵ 
	// true : Use HSV interpolation
	// false : Use RGB interpolation
	bool		m_bUseHSVInterpolation;

public:
	// 
	virtual void GenDirection(A3DVECTOR3& vDir);
	// 
	virtual float GenTTL();
	// ɫ
	virtual A3DCOLOR GenColor();
	// ӲĿ
	virtual int GenEmissionCount(float fTimeSpan);
	// ӱ
	virtual float GenScale();
	// ӳʼǶ
	virtual float GenRotAngle();

public:
	static A3DParticleEmitter* CreateEmitter(int nType);
	virtual int GetEmissionCount(float fTimeSpan) = 0;
	virtual void InitParticle(A3DParticle* pParticle)
	{
		pParticle->m_fVelAlongAcc	= 0;
		pParticle->m_fScaleNoise	= 0;
		pParticle->m_color			= GenColor();
		pParticle->m_fScale			= GenScale();
		pParticle->m_fRotAngle		= GenRotAngle();
		pParticle->m_fSelfVel		= m_fSpeed;
		pParticle->m_vDir			= _unit_dir;
		pParticle->m_vAxisOff.Clear();
	}
	void SetScale(float fMin, float fMax) { m_fScaleMin = fMin; m_fScaleMax = fMax; }
	float GetScaleMin() const { return m_fScaleMin; }
	float GetScaleMax() const { return m_fScaleMax; }
	void SetParentDirAndPos(const A3DVECTOR3& vPos, const A3DQUATERNION& qDir)
	{
		m_vParentPos = vPos;
		m_qParentDir = qDir;
		calcParticleDir();
	}
	virtual void UpdateEmitterScale(float fScale) {}
	virtual int CalcPoolSize() const { return (int)(m_fTTL * m_fEmissionRate) + 1; }
	virtual void CloneFrom(const A3DParticleEmitter* p);
	float GetSpeed() const { return m_fSpeed; }
	float GetParAcc() const { return m_fParAcc; }
	A3DVECTOR3 GetAccDir() const { return m_vAccDir; }
	float GetAcc() const { return m_fAcc; }
	bool IsBind() const { return m_bBind; }
	bool IsDrag() const { return m_bDrag; }
	float GetDragIntensity() const { return m_fDragIntensity; }
	void UpdateEmitterAlpha(float fAlpha) { m_fEmitterAlpha = fAlpha; }
	void SetEmissionRate(float fRate) { m_fEmissionRate = fRate; }
	void SetEmissionAng(float fAngle) { m_fAngle = fAngle; }
	void SetSpeed(float fSpeed) { m_fSpeed = fSpeed; }
	void SetTTL(float fTTL) { m_fTTL = fTTL; }
	A3DGFXKeyPoint* GetKeyPoint() { return m_pKeyPoint; }

public:
	virtual bool SetProperty(int nOp, const GFX_PROPERTY& prop)
	{
		switch (nOp)
		{
		case ID_GFXOP_EMITTER_RATE:
			m_fEmissionRate = prop;
			break;
		case ID_GFXOP_EMITTER_ANGLE:
			m_fAngle = DEG2RAD((float)prop);
			break;
		case ID_GFXOP_EMITTER_SPEED:
			m_fSpeed = prop;
			break;
		case ID_GFXOP_EMITTER_PAR_ACC:
			m_fParAcc = prop;
			break;
		case ID_GFXOP_EMITTER_ACC_DIR:
			m_vAccDir = prop;
			m_vAccDir.Normalize();
			break;
		case ID_GFXOP_EMITTER_ACC:
			m_fAcc = prop;
			break;
		case ID_GFXOP_EMITTER_TTL:
			m_fTTL = prop;
			m_fTTL /= 1000.f;
			break;
		case ID_GFXOP_EMITTER_COLOR_MIN:
			m_colorMin = prop;
			a3d_RGBToHSV(m_colorMin, &m_hsvMin);
			break;
		case ID_GFXOP_EMITTER_COLOR_MAX:
			m_colorMax = prop;
			a3d_RGBToHSV(m_colorMax, &m_hsvMax);
			break;
		case ID_GFXOP_EMITTER_SCALE_MIN:
			m_fScaleMin = prop;
			break;
		case ID_GFXOP_EMITTER_SCALE_MAX:
			m_fScaleMax = prop;
			break;
		case ID_GFXOP_EMITTER_ROT_MIN:
			m_fRotMin = DEG2RAD((float)prop);
			break;
		case ID_GFXOP_EMITTER_ROT_MAX:
			m_fRotMax = DEG2RAD((float)prop);
			break;
		case ID_GFXOP_EMITTER_SURFACE:
			m_bSurface = prop;
			break;
		case ID_GFXOP_EMITTER_BIND:
			m_bBind = prop;
			break;
		case ID_GFXOP_EMITTER_DRAG:
			m_bDrag = prop;
			break;
		case ID_GFXOP_EMITTER_DRAG_POWER:
			m_fDragIntensity = prop;
			break;
		case ID_GFXOP_EMITTER_PARTICAL_INITDIR:
			{
				A3DVECTOR3 vTmp = (A3DVECTOR3)prop;
				m_vParticalInitDir.Set(DEG2RAD(vTmp.x), DEG2RAD(vTmp.y), DEG2RAD(vTmp.z));
				calcParticleDir();
			}
			break;
		case ID_GFXOP_EMITTER_COLOR_HSVINTERP:
			m_bUseHSVInterpolation = prop;
			break;
		}
		return true;
	}
	virtual GFX_PROPERTY GetProperty(int nOp) const
	{
		switch (nOp)
		{
		case ID_GFXOP_EMITTER_RATE:
			return GFX_PROPERTY(m_fEmissionRate);
		case ID_GFXOP_EMITTER_ANGLE:
			return GFX_PROPERTY(RAD2DEG(m_fAngle));
		case ID_GFXOP_EMITTER_SPEED:
			return GFX_PROPERTY(m_fSpeed);
		case ID_GFXOP_EMITTER_PAR_ACC:
			return GFX_PROPERTY(m_fParAcc);
		case ID_GFXOP_EMITTER_ACC_DIR:
			return GFX_PROPERTY(m_vAccDir);
		case ID_GFXOP_EMITTER_ACC:
			return GFX_PROPERTY(m_fAcc);
		case ID_GFXOP_EMITTER_TTL:
			return GFX_PROPERTY(m_fTTL * 1000.f);
		case ID_GFXOP_EMITTER_COLOR_MIN:
			return GFX_PROPERTY(m_colorMin);
		case ID_GFXOP_EMITTER_COLOR_MAX:
			return GFX_PROPERTY(m_colorMax);
		case ID_GFXOP_EMITTER_SCALE_MIN:
			return GFX_PROPERTY(m_fScaleMin);
		case ID_GFXOP_EMITTER_SCALE_MAX:
			return GFX_PROPERTY(m_fScaleMax);
		case ID_GFXOP_EMITTER_ROT_MIN:
			return GFX_PROPERTY(RAD2DEG(m_fRotMin));
		case ID_GFXOP_EMITTER_ROT_MAX:
			return GFX_PROPERTY(RAD2DEG(m_fRotMax));
		case ID_GFXOP_EMITTER_SURFACE:
			return GFX_PROPERTY(m_bSurface);
		case ID_GFXOP_EMITTER_BIND:
			return GFX_PROPERTY(m_bBind);
		case ID_GFXOP_EMITTER_DRAG:
			return GFX_PROPERTY(m_bDrag);
		case ID_GFXOP_EMITTER_DRAG_POWER:
			return GFX_PROPERTY(m_fDragIntensity);
		case ID_GFXOP_EMITTER_PARTICAL_INITDIR:
			return GFX_PROPERTY(A3DVECTOR3(RAD2DEG(m_vParticalInitDir.x), RAD2DEG(m_vParticalInitDir.y), RAD2DEG(m_vParticalInitDir.z)));
		case ID_GFXOP_EMITTER_COLOR_HSVINTERP:
			return GFX_PROPERTY(m_bUseHSVInterpolation);
		}
		return GFX_PROPERTY();
	}

	virtual bool Load(AFile* pFile, DWORD dwVersion);
	virtual bool Save(AFile* pFile);
	virtual void Resume()
	{
		m_fRemainder = 0.0f;
	}

protected:
	inline void calcParticleDir()
	{ m_qParticleDir = m_qParentDir * A3DQUATERNION(m_vParticalInitDir.x, m_vParticalInitDir.y, m_vParticalInitDir.z); }
};

#endif
